<?php

namespace App\Http\Controllers;

use App\Models\AlertRule;
use App\Models\CaseItem;
use App\Models\CaseOption;
use App\Models\Client;
use App\Models\User;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;
use Illuminate\Validation\Rule;

class CaseController extends Controller
{
    public function index(Request $request)
    {
        $user = $request->user();

        $clientsQuery = $this->visibleClientsQuery($request);
        $visibleClientIds = $clientsQuery->pluck('clients.id');

        $query = CaseItem::query()->whereIn('client_id', $visibleClientIds);

        if ($user->hasRole('Client Viewer')) {
            $query->where('client_id', $user->client_id);
        }

        if ($request->filled('client_id') && !$user->hasRole('Client Viewer')) {
            $query->where('client_id', (int) $request->input('client_id'));
        }

        foreach (['label', 'source', 'sentiment', 'status'] as $field) {
            if ($request->filled($field)) {
                $query->where($field, $request->input($field));
            }
        }

        if ($request->filled('created_by')) {
            $query->where('created_by', (int) $request->input('created_by'));
        }

        if ($request->filled('date_from')) {
            $query->whereDate('post_date', '>=', $request->input('date_from'));
        }

        if ($request->filled('date_to')) {
            $query->whereDate('post_date', '<=', $request->input('date_to'));
        }

        [$thresholdHours, $notClosedStatuses] = $this->slaConfig();

        if ($request->boolean('overdue_only')) {
            $query->whereIn('status', $notClosedStatuses)
                ->where('created_at', '<=', now()->subHours($thresholdHours));
        }

        $cases = $query
            ->with(['client', 'creator', 'owners'])
            ->orderByDesc('created_at')
            ->paginate(20)
            ->withQueryString();

        $labelClientId = null;
        if ($user->hasRole('Client Viewer')) {
            $labelClientId = $user->client_id;
        } elseif ($request->filled('client_id')) {
            $labelClientId = (int) $request->input('client_id');
        }

        $options = $this->options($labelClientId);
        $labelMap = $this->labelMapForClients($visibleClientIds->all());

        return view('cases.index', [
            'cases' => $cases,
            'clients' => $clientsQuery->orderBy('name')->get(),
            'options' => $options,
            'sla_threshold_hours' => $thresholdHours,
            'not_closed_statuses' => $notClosedStatuses,
            'labelMap' => $labelMap,
        ]);
    }

    public function create(Request $request)
    {
        $user = $request->user();

        abort_unless($user->can('cases.create'), 403);

        $clients = $this->visibleClientsQuery($request)->orderBy('name')->get();
        $firstClientId = $clients->first()?->id;
        $options = $this->options($firstClientId);
        $labelMap = $this->labelMapForClients($clients->pluck('id')->all());

        return view('cases.create', [
            'clients' => $clients,
            'options' => $options,
            'labelMap' => $labelMap,
        ]);
    }

    public function store(Request $request)
    {
        $user = $request->user();

        abort_unless($user->can('cases.create'), 403);

        $visibleClientIds = $this->visibleClientsQuery($request)
            ->pluck('clients.id')
            ->map(fn ($id) => (int) $id)
            ->all();
        $postedClientId = (int) $request->input('client_id');
        if (!in_array($postedClientId, $visibleClientIds, true) && count($visibleClientIds) === 1) {
            $request->merge(['client_id' => $visibleClientIds[0]]);
        }
        $this->normalizeDateTime($request);

        $options = $this->options();

        $data = $request->validate([
            'client_id' => ['required', 'integer', Rule::in($visibleClientIds)],
            'label' => ['required', 'string', Rule::in($options['label'])],
            'source' => ['required', 'string', Rule::in($options['source'])],
            'sentiment' => ['required', 'string', Rule::in($options['sentiment'])],
            'post_date' => ['required', 'date'],
            'post_time' => ['nullable', 'date_format:H:i'],
            'post_text' => ['nullable', 'string'],
            'translation' => ['nullable', 'string'],
            'post_link' => ['nullable', 'string', 'max:2048'],
            'action_social' => ['nullable', 'string', Rule::in($options['action_social'])],
            'notes_social' => ['nullable', 'string'],
            'user_reply' => ['nullable', 'string'],
            'status' => ['required', 'string', Rule::in($options['status'])],
        ]);

        abort_unless($user->can('create', [CaseItem::class, (int) $data['client_id']]), 403);

        $data['created_by'] = $user->id;
        $data['last_status_changed_at'] = now();

        if ($data['status'] === 'Closed') {
            $data['closed_at'] = now();
        }

        $case = CaseItem::create($data);

        return redirect()->route('cases.show', $case)->with('status', 'Case created.');
    }

    public function show(Request $request, CaseItem $case)
    {
        $this->authorize('view', $case);

        $case->load(['client', 'creator', 'owners', 'comments.user']);

        $user = $request->user();
        $ownerCandidates = User::query()
            ->whereDoesntHave('roles', fn ($q) => $q->where('name', 'Client Viewer'))
            ->when(!$user->hasRole('Super Admin'), function ($query) use ($case) {
                $query->whereHas('assignedClients', fn ($q) => $q->where('clients.id', $case->client_id));
            })
            ->orderBy('name')
            ->get();

        $activities = DB::table('activity_log')
            ->leftJoin('users', 'users.id', '=', 'activity_log.causer_id')
            ->where(function ($q) use ($case) {
                $q->where('subject_type', CaseItem::class)
                    ->where('subject_id', $case->id);
            })
            ->select('activity_log.*', 'users.name as causer_name')
            ->orderByDesc('id')
            ->limit(200)
            ->get();

        $options = $this->options($case->client_id);

        return view('cases.show', [
            'case' => $case,
            'activities' => $activities,
            'options' => $options,
            'ownerCandidates' => $ownerCandidates,
        ]);
    }

    public function edit(Request $request, CaseItem $case)
    {
        $this->authorize('update', $case);

        $clients = $this->visibleClientsQuery($request)->orderBy('name')->get();
        $options = $this->options($case->client_id);
        $labelMap = $this->labelMapForClients($clients->pluck('id')->all());

        return view('cases.edit', [
            'case' => $case,
            'clients' => $clients,
            'options' => $options,
            'labelMap' => $labelMap,
        ]);
    }

    public function update(Request $request, CaseItem $case)
    {
        $this->authorize('update', $case);

        $user = $request->user();
        $visibleClientIds = $this->visibleClientsQuery($request)
            ->pluck('clients.id')
            ->map(fn ($id) => (int) $id)
            ->all();
        $visibleClientIds[] = (int) $case->client_id;
        $visibleClientIds = array_values(array_unique($visibleClientIds));
        $postedClientId = (int) $request->input('client_id');
        if (!in_array($postedClientId, $visibleClientIds, true) && count($visibleClientIds) === 1) {
            $request->merge(['client_id' => $visibleClientIds[0]]);
        }
        $this->normalizeDateTime($request);
        $options = $this->options();

        $data = $request->validate([
            'client_id' => ['required', 'integer', Rule::in($visibleClientIds)],
            'label' => ['required', 'string', Rule::in($options['label'])],
            'source' => ['required', 'string', Rule::in($options['source'])],
            'sentiment' => ['required', 'string', Rule::in($options['sentiment'])],
            'post_date' => ['required', 'date'],
            'post_time' => ['nullable', 'date_format:H:i'],
            'post_text' => ['nullable', 'string'],
            'translation' => ['nullable', 'string'],
            'post_link' => ['nullable', 'string', 'max:2048'],
            'action_social' => ['nullable', 'string', Rule::in($options['action_social'])],
            'notes_social' => ['nullable', 'string'],
            'user_reply' => ['nullable', 'string'],
            'status' => ['required', 'string', Rule::in($options['status'])],
        ]);

        $case->fill($data);

        if ($case->isDirty('status')) {
            $case->last_status_changed_at = now();
            if ($case->status === 'Closed' && $case->closed_at === null) {
                abort_unless(
                    $user->hasAnyRole(['Super Admin', 'Account Director', 'Community Team Member', 'Listening Team Member']),
                    403
                );
                $case->closed_at = now();
            }
            if ($case->status !== 'Closed') {
                $case->closed_at = null;
            }
        }

        $case->save();

        return redirect()->route('cases.show', $case)->with('status', 'Case updated.');
    }

    public function destroy(Request $request, CaseItem $case)
    {
        $this->authorize('update', $case);

        $case->delete();

        return redirect()->route('cases.index')->with('status', 'Case deleted.');
    }

    public function bulkDestroy(Request $request)
    {
        $ids = $request->validate([
            'ids' => ['required', 'array', 'min:1'],
            'ids.*' => ['integer', 'exists:cases,id'],
        ])['ids'];

        $cases = CaseItem::query()->whereIn('id', $ids)->get();
        foreach ($cases as $case) {
            $this->authorize('update', $case);
            $case->delete();
        }

        return redirect()->route('cases.index')->with('status', 'Selected cases deleted.');
    }

    private function visibleClientsQuery(Request $request)
    {
        $user = $request->user();

        if ($user->hasRole('Super Admin')) {
            return Client::query();
        }

        if ($user->hasRole('Client Viewer')) {
            return Client::query()->whereKey($user->client_id);
        }

        return $user->assignedClients();
    }

    private function options(?int $clientId = null): array
    {
        $groups = ['label', 'source', 'sentiment', 'action_social', 'status'];
        $out = [];

        foreach ($groups as $group) {
            if ($group === 'label') {
                $out[$group] = $this->labelOptionsForClient($clientId);
                continue;
            }

            $out[$group] = CaseOption::query()
                ->where('group', $group)
                ->where('is_active', true)
                ->orderBy('sort_order')
                ->pluck('value')
                ->all();
        }

        return $out;
    }

    private function labelOptionsForClient(?int $clientId = null): array
    {
        $allLabels = CaseOption::query()
            ->where('group', 'label')
            ->where('is_active', true)
            ->orderBy('sort_order')
            ->pluck('value', 'id')
            ->all();

        if (!$clientId) {
            return array_values($allLabels);
        }

        $client = Client::query()->with('labelOptions')->find($clientId);
        if (!$client) {
            return array_values($allLabels);
        }

        $assigned = $client->labelOptions->pluck('value')->all();
        if (count($assigned) === 0) {
            return array_values($allLabels);
        }

        return $assigned;
    }

    private function labelMapForClients(array $clientIds): array
    {
        $map = [];
        foreach ($clientIds as $clientId) {
            $map[$clientId] = $this->labelOptionsForClient((int) $clientId);
        }

        return $map;
    }

    private function slaConfig(): array
    {
        $rule = AlertRule::query()
            ->where('enabled', true)
            ->where('is_global', true)
            ->orderByDesc('id')
            ->first();

        $thresholdHours = (int) ($rule?->threshold_hours ?? 24);
        $notClosedStatuses = $rule?->statuses;

        if (!is_array($notClosedStatuses) || count($notClosedStatuses) === 0) {
            $notClosedStatuses = ['Open', 'In Progress', 'Waiting for User', 'Resolved'];
        }

        return [$thresholdHours, $notClosedStatuses];
    }

    private function normalizeDateTime(Request $request): void
    {
        if ($request->filled('post_date')) {
            try {
                $request->merge([
                    'post_date' => \Illuminate\Support\Carbon::parse((string) $request->input('post_date'))->format('Y-m-d'),
                ]);
            } catch (\Throwable $e) {
                // Keep original value for validator error messaging.
            }
        }

        if ($request->filled('post_time')) {
            try {
                $request->merge([
                    'post_time' => \Illuminate\Support\Carbon::parse((string) $request->input('post_time'))->format('H:i'),
                ]);
            } catch (\Throwable $e) {
                // Keep original value for validator error messaging.
            }
        }
    }
}
