<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Models\CaseOption;
use App\Models\Client;
use Illuminate\Http\Request;

class ClientController extends Controller
{
    public function index(Request $request)
    {
        abort_unless($request->user()->can('clients.manage'), 403);

        $clients = Client::query()->orderBy('name')->paginate(20);

        return view('admin.clients.index', [
            'clients' => $clients,
        ]);
    }

    public function create(Request $request)
    {
        abort_unless($request->user()->can('clients.manage'), 403);

        $labels = CaseOption::query()
            ->where('group', 'label')
            ->where('is_active', true)
            ->orderBy('sort_order')
            ->get();

        return view('admin.clients.create', [
            'labels' => $labels,
        ]);
    }

    public function store(Request $request)
    {
        abort_unless($request->user()->can('clients.manage'), 403);

        $data = $request->validate([
            'name' => ['required', 'string', 'max:255'],
            'status' => ['required', 'in:active,inactive'],
            'timezone' => ['required', 'string', 'max:100'],
            'default_notification_emails' => ['nullable', 'string'],
            'label_ids' => ['array'],
            'label_ids.*' => ['integer', 'exists:case_options,id'],
        ]);

        $emails = collect(explode(',', (string) ($data['default_notification_emails'] ?? '')))
            ->map(fn ($e) => trim($e))
            ->filter()
            ->values()
            ->all();

        $client = Client::create([
            'name' => $data['name'],
            'status' => $data['status'],
            'timezone' => $data['timezone'],
            'default_notification_emails' => $emails,
        ]);

        $client->labelOptions()->sync($data['label_ids'] ?? []);

        return redirect()->route('admin.clients.edit', $client)->with('status', 'Client created.');
    }

    public function edit(Request $request, Client $client)
    {
        abort_unless($request->user()->can('clients.manage'), 403);

        $client->load('labelOptions');

        return view('admin.clients.edit', [
            'client' => $client,
            'labels' => CaseOption::query()
                ->where('group', 'label')
                ->where('is_active', true)
                ->orderBy('sort_order')
                ->get(),
        ]);
    }

    public function update(Request $request, Client $client)
    {
        abort_unless($request->user()->can('clients.manage'), 403);

        $data = $request->validate([
            'name' => ['required', 'string', 'max:255'],
            'status' => ['required', 'in:active,inactive'],
            'timezone' => ['required', 'string', 'max:100'],
            'default_notification_emails' => ['nullable', 'string'],
            'label_ids' => ['array'],
            'label_ids.*' => ['integer', 'exists:case_options,id'],
        ]);

        $emails = collect(explode(',', (string) ($data['default_notification_emails'] ?? '')))
            ->map(fn ($e) => trim($e))
            ->filter()
            ->values()
            ->all();

        $client->update([
            'name' => $data['name'],
            'status' => $data['status'],
            'timezone' => $data['timezone'],
            'default_notification_emails' => $emails,
        ]);

        $client->labelOptions()->sync($data['label_ids'] ?? []);

        return redirect()->route('admin.clients.edit', $client)->with('status', 'Client updated.');
    }

    public function destroy(Request $request, Client $client)
    {
        abort_unless($request->user()->can('clients.manage'), 403);

        $client->delete();

        return redirect()->route('admin.clients.index')->with('status', 'Client deleted.');
    }

    public function bulkDestroy(Request $request)
    {
        abort_unless($request->user()->can('clients.manage'), 403);

        $ids = $request->validate([
            'ids' => ['required', 'array', 'min:1'],
            'ids.*' => ['integer', 'exists:clients,id'],
        ])['ids'];

        Client::query()->whereIn('id', $ids)->delete();

        return redirect()->route('admin.clients.index')->with('status', 'Selected clients deleted.');
    }
}
