<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Models\Client;
use App\Models\User;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Hash;
use Spatie\Permission\Models\Role;

class UserController extends Controller
{
    public function index(Request $request)
    {
        abort_unless($request->user()->can('users.manage'), 403);

        $users = User::query()
            ->with(['roles', 'client'])
            ->orderBy('name')
            ->paginate(25);

        return view('admin.users.index', [
            'users' => $users,
        ]);
    }

    public function create(Request $request)
    {
        abort_unless($request->user()->can('users.manage'), 403);

        return view('admin.users.create', [
            'roles' => Role::query()->orderBy('name')->get(),
            'clients' => Client::query()->orderBy('name')->get(),
        ]);
    }

    public function store(Request $request)
    {
        abort_unless($request->user()->can('users.manage'), 403);

        $data = $request->validate([
            'name' => ['required', 'string', 'max:255'],
            'email' => ['required', 'email', 'max:255', 'unique:users,email'],
            'password' => ['required', 'string', 'min:8'],
            'is_active' => ['nullable', 'boolean'],
            'role' => ['required', 'string', 'exists:roles,name'],
            'client_id' => ['nullable', 'integer', 'exists:clients,id'],
            'assigned_client_ids' => ['array'],
            'assigned_client_ids.*' => ['integer', 'exists:clients,id'],
        ]);

        $user = User::create([
            'name' => $data['name'],
            'email' => $data['email'],
            'password' => Hash::make($data['password']),
            'is_active' => (bool) ($data['is_active'] ?? true),
            'client_id' => $data['client_id'] ?? null,
        ]);

        $user->syncRoles([$data['role']]);

        $assignedIds = $data['assigned_client_ids'] ?? [];
        $user->assignedClients()->sync($assignedIds);

        return redirect()->route('admin.users.edit', $user)->with('status', 'User created.');
    }

    public function edit(Request $request, User $user)
    {
        abort_unless($request->user()->can('users.manage'), 403);

        $user->load(['roles', 'assignedClients']);

        return view('admin.users.edit', [
            'user' => $user,
            'roles' => Role::query()->orderBy('name')->get(),
            'clients' => Client::query()->orderBy('name')->get(),
        ]);
    }

    public function update(Request $request, User $user)
    {
        abort_unless($request->user()->can('users.manage'), 403);

        $data = $request->validate([
            'name' => ['required', 'string', 'max:255'],
            'email' => ['required', 'email', 'max:255', 'unique:users,email,' . $user->id],
            'password' => ['nullable', 'string', 'min:8'],
            'is_active' => ['nullable', 'boolean'],
            'role' => ['required', 'string', 'exists:roles,name'],
            'client_id' => ['nullable', 'integer', 'exists:clients,id'],
            'assigned_client_ids' => ['array'],
            'assigned_client_ids.*' => ['integer', 'exists:clients,id'],
        ]);

        $user->name = $data['name'];
        $user->email = $data['email'];
        $user->is_active = (bool) ($data['is_active'] ?? false);
        $user->client_id = $data['client_id'] ?? null;

        if (!empty($data['password'])) {
            $user->password = Hash::make($data['password']);
        }

        $user->save();

        $user->syncRoles([$data['role']]);

        $assignedIds = $data['assigned_client_ids'] ?? [];
        $user->assignedClients()->sync($assignedIds);

        return redirect()->route('admin.users.edit', $user)->with('status', 'User updated.');
    }

    public function destroy(Request $request, User $user)
    {
        abort_unless($request->user()->can('users.manage'), 403);

        $user->delete();

        return redirect()->route('admin.users.index')->with('status', 'User deleted.');
    }

    public function bulkDestroy(Request $request)
    {
        abort_unless($request->user()->can('users.manage'), 403);

        $ids = $request->validate([
            'ids' => ['required', 'array', 'min:1'],
            'ids.*' => ['integer', 'exists:users,id'],
        ])['ids'];

        User::query()->whereIn('id', $ids)->delete();

        return redirect()->route('admin.users.index')->with('status', 'Selected users deleted.');
    }
}
