<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Models\CaseOption;
use Illuminate\Http\Request;
use Illuminate\Validation\Rule;

class LabelController extends Controller
{
    public function index(Request $request)
    {
        abort_unless($request->user()->can('clients.manage'), 403);

        $labels = CaseOption::query()
            ->where('group', 'label')
            ->orderBy('sort_order')
            ->paginate(25);

        return view('admin.labels.index', [
            'labels' => $labels,
        ]);
    }

    public function create(Request $request)
    {
        abort_unless($request->user()->can('clients.manage'), 403);

        return view('admin.labels.create');
    }

    public function store(Request $request)
    {
        abort_unless($request->user()->can('clients.manage'), 403);

        $data = $request->validate([
            'value' => ['required', 'string', 'max:255', Rule::unique('case_options', 'value')->where('group', 'label')],
            'sort_order' => ['nullable', 'integer', 'min:0'],
            'is_active' => ['nullable', 'boolean'],
        ]);

        CaseOption::create([
            'group' => 'label',
            'value' => $data['value'],
            'sort_order' => $data['sort_order'] ?? 0,
            'is_active' => (bool) ($data['is_active'] ?? true),
        ]);

        return redirect()->route('admin.labels.index')->with('status', 'Label added.');
    }

    public function edit(Request $request, CaseOption $label)
    {
        abort_unless($request->user()->can('clients.manage'), 403);

        if ($label->group !== 'label') {
            abort(404);
        }

        return view('admin.labels.edit', [
            'label' => $label,
        ]);
    }

    public function update(Request $request, CaseOption $label)
    {
        abort_unless($request->user()->can('clients.manage'), 403);

        if ($label->group !== 'label') {
            abort(404);
        }

        $data = $request->validate([
            'value' => ['required', 'string', 'max:255', Rule::unique('case_options', 'value')->where('group', 'label')->ignore($label->id)],
            'sort_order' => ['nullable', 'integer', 'min:0'],
            'is_active' => ['nullable', 'boolean'],
        ]);

        $label->update([
            'value' => $data['value'],
            'sort_order' => $data['sort_order'] ?? $label->sort_order,
            'is_active' => (bool) ($data['is_active'] ?? false),
        ]);

        return redirect()->route('admin.labels.edit', $label)->with('status', 'Label updated.');
    }

    public function destroy(Request $request, CaseOption $label)
    {
        abort_unless($request->user()->can('clients.manage'), 403);

        if ($label->group !== 'label') {
            abort(404);
        }

        $label->delete();

        return redirect()->route('admin.labels.index')->with('status', 'Label deleted.');
    }

    public function bulkDestroy(Request $request)
    {
        abort_unless($request->user()->can('clients.manage'), 403);

        $ids = $request->validate([
            'ids' => ['required', 'array', 'min:1'],
            'ids.*' => ['integer', 'exists:case_options,id'],
        ])['ids'];

        CaseOption::query()->where('group', 'label')->whereIn('id', $ids)->delete();

        return redirect()->route('admin.labels.index')->with('status', 'Selected labels deleted.');
    }
}
