<?php

namespace App\Http\Controllers;

use App\Models\AlertRule;
use App\Models\CaseItem;
use App\Models\Client;
use App\Models\ReportDispatchLog;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;

class DashboardController extends Controller
{
    public function __invoke(Request $request)
    {
        $user = $request->user();

        if ($user->hasRole('Super Admin')) {
            return $this->admin($request);
        }

        if ($user->hasRole('Client Viewer')) {
            return $this->client($request);
        }

        return $this->agency($request);
    }

    private function admin(Request $request)
    {
        $clientIds = Client::query()->pluck('id');

        [$kpis, $charts] = $this->buildDashboardData($clientIds);

        return view('dashboards.admin', [
            'kpis' => $kpis,
            'charts' => $charts,
            'reportLogs' => ReportDispatchLog::query()->with('rule')->latest('ran_at')->limit(15)->get(),
        ]);
    }

    private function agency(Request $request)
    {
        $clientIds = $request->user()->assignedClients()->pluck('clients.id');

        [$kpis, $charts] = $this->buildDashboardData($clientIds);

        return view('dashboards.agency', [
            'kpis' => $kpis,
            'charts' => $charts,
            'clients' => $request->user()->assignedClients()->orderBy('name')->get(),
        ]);
    }

    private function client(Request $request)
    {
        $clientIds = collect([$request->user()->client_id]);

        [$kpis, $charts] = $this->buildDashboardData($clientIds);

        return view('dashboards.client', [
            'kpis' => $kpis,
            'charts' => $charts,
        ]);
    }

    private function buildDashboardData($clientIds): array
    {
        $rule = AlertRule::query()
            ->where('enabled', true)
            ->where('is_global', true)
            ->orderByDesc('id')
            ->first();

        $thresholdHours = (int) ($rule?->threshold_hours ?? 24);
        $notClosedStatuses = $rule?->statuses;

        if (!is_array($notClosedStatuses) || count($notClosedStatuses) === 0) {
            $notClosedStatuses = ['Open', 'In Progress', 'Waiting for User', 'Resolved'];
        }

        $base = CaseItem::query()->whereIn('client_id', $clientIds);

        $kpis = [
            'clients' => (int) $clientIds->count(),
            'total_cases' => (clone $base)->count(),
            'critical_cases' => (clone $base)->where('sentiment', 'Critical')->where('status', '!=', 'Closed')->count(),
            'overdue_cases' => (clone $base)
                ->whereIn('status', $notClosedStatuses)
                ->where('created_at', '<=', now()->subHours($thresholdHours))
                ->count(),
            'by_status' => (clone $base)
                ->select('status', DB::raw('COUNT(*) as cnt'))
                ->groupBy('status')
                ->pluck('cnt', 'status')
                ->all(),
        ];

        $trend = (clone $base)
            ->where('created_at', '>=', now()->subDays(14))
            ->select(DB::raw('DATE(created_at) as day'), DB::raw('COUNT(*) as cnt'))
            ->groupBy('day')
            ->orderBy('day')
            ->get();

        $sentiment = (clone $base)
            ->select('sentiment', DB::raw('COUNT(*) as cnt'))
            ->groupBy('sentiment')
            ->orderByDesc('cnt')
            ->get();

        $labels = (clone $base)
            ->select('label', DB::raw('COUNT(*) as cnt'))
            ->groupBy('label')
            ->orderByDesc('cnt')
            ->limit(10)
            ->get();

        $sources = (clone $base)
            ->select('source', DB::raw('COUNT(*) as cnt'))
            ->groupBy('source')
            ->orderByDesc('cnt')
            ->get();

        $charts = [
            'trend_labels' => $trend->pluck('day')->all(),
            'trend_counts' => $trend->pluck('cnt')->all(),
            'sentiment_labels' => $sentiment->pluck('sentiment')->all(),
            'sentiment_counts' => $sentiment->pluck('cnt')->all(),
            'label_labels' => $labels->pluck('label')->all(),
            'label_counts' => $labels->pluck('cnt')->all(),
            'source_labels' => $sources->pluck('source')->all(),
            'source_counts' => $sources->pluck('cnt')->all(),
            'sla_threshold_hours' => $thresholdHours,
        ];

        return [$kpis, $charts];
    }
}
