<?php

namespace App\Http\Controllers;

use App\Models\CaseItem;
use App\Models\Client;
use Illuminate\Http\Request;
use Symfony\Component\HttpFoundation\StreamedResponse;

class CaseExportController extends Controller
{
    public function __invoke(Request $request): StreamedResponse
    {
        abort_unless($request->user()->can('cases.export'), 403);

        $user = $request->user();

        $clientIds = $this->visibleClientsQuery($request)->pluck('clients.id');
        $query = CaseItem::query()->whereIn('client_id', $clientIds);

        if ($user->hasRole('Client Viewer')) {
            $query->where('client_id', $user->client_id);
        }

        foreach (['client_id', 'label', 'source', 'sentiment', 'status', 'created_by'] as $field) {
            if ($request->filled($field)) {
                $query->where($field, $request->input($field));
            }
        }

        if ($request->filled('date_from')) {
            $query->whereDate('post_date', '>=', $request->input('date_from'));
        }

        if ($request->filled('date_to')) {
            $query->whereDate('post_date', '<=', $request->input('date_to'));
        }

        $filename = 'cases_export_' . now()->format('Ymd_His') . '.csv';

        return response()->streamDownload(function () use ($query) {
            $out = fopen('php://output', 'w');

            fputcsv($out, [
                'id',
                'client_id',
                'label',
                'source',
                'sentiment',
                'post_date',
                'post_time',
                'post_link',
                'status',
                'created_by',
                'created_at',
                'updated_at',
                'closed_at',
            ]);

            $query->orderBy('id')->chunk(500, function ($cases) use ($out) {
                foreach ($cases as $case) {
                    fputcsv($out, [
                        $case->id,
                        $case->client_id,
                        $case->label,
                        $case->source,
                        $case->sentiment,
                        optional($case->post_date)->format('Y-m-d'),
                        $case->post_time,
                        $case->post_link,
                        $case->status,
                        $case->created_by,
                        optional($case->created_at)->toDateTimeString(),
                        optional($case->updated_at)->toDateTimeString(),
                        optional($case->closed_at)->toDateTimeString(),
                    ]);
                }
            });

            fclose($out);
        }, $filename, [
            'Content-Type' => 'text/csv; charset=UTF-8',
        ]);
    }

    private function visibleClientsQuery(Request $request)
    {
        $user = $request->user();

        if ($user->hasRole('Super Admin')) {
            return Client::query();
        }

        if ($user->hasRole('Client Viewer')) {
            return Client::query()->whereKey($user->client_id);
        }

        return $user->assignedClients()->getQuery();
    }
}
